 /**
 *
 * \file        HPRFGW_RFResultCode.c
 *
 * \brief       This file contains the RF result code timer and processing functions		
 *
 * \detail		
 *
 * \author      Hazrat Shah	
 *
 * \date        10/10/2006
 * \warning	
 * \note
 * \review
 */

#include "hprfgw_config.h"
#include "hprfgw_rf802_15_4.h"
#include "hprfgw_rfDesc.h"
#include "hprfgw_rfApp.h"
#include "hprfgw_rfComLog.h"
#include "hprfgw_rfTrans.h"
#include "hprfgw_rfRec.h"
#include "hprfgw_rfHeartbeat.h"
#include "hprfgw_rfResultCode.h"
#include "hprfgw_rfSlaveIntToHost.h"
#include "hprfgw_rfDiagnostics.h"

////////////////////////////////////////////////////////////////////////////////
//	DEFINITIONS	
////////////////////////////////////////////////////////////////////////////////
#define	RF_RESULT_CODE_TASK_RATE		(1000)		//	Rate is 10-seconds in milli-second
#define	RF_RESULT_CODE_TASK_STACK_SIZE	(8*1024)	//	Task stack
#ifndef RF_TASKS_BASE_PRIORITY
#if defined (OS_NUCLEUS)
 #pragma info "!!!!Pass in the RF_TASKS_BASE_PRIORITY using the make file!!!!"
#endif
#endif
#define RF_RESULT_CODE_TASK_PRIORITY	RF_TASK_PRIORITY(RF_TASKS_BASE_PRIORITY, 2)		//

#define	RF_RESET_PIPE_ON_RESULT_CODE_TIMEOUT_ENABLED	
////////////////////////////////////////////////////////////////////////////////
//	LOCAL DATA		
////////////////////////////////////////////////////////////////////////////////
typedef struct	{
#if defined (OS_NUCLEUS)
   	NU_TASK 	task;
	UINT8   	stack[RF_RESULT_CODE_TASK_STACK_SIZE];
#elif defined (OS_WINCE)
	HANDLE		task;
#endif
	DM_RESOURCE_HANDLE task;
}	TYPE_RFRESULTCODE_TASK;
TYPE_RFRESULTCODE_TASK RFResultCodeTask;

#ifdef	RF_DBG_LOG
	void RFDebugLogShow (UINT32 entries);
#endif

////////////////////////////////////////////////////////////////////////////////
//	
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
//	LOCAL FUNCTIONS 
///////////////////////////////////////////////////////////////////////////////

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		The RF Result Code counter service task  
 *
 * \detail		This functions serivces the result code timer of all rf devices whose link state is either active or Tempconn
 *				If the result code timer expires the device "Result code Timedout" flag and Result code timeout event are set.
 *				The rf receiver task waits on the result code timeout event.		
 *
 * \date        10/12/2006
 *
 * \param	    argc, argv		
 *
 * \return		None	 				
 *
 * \retval		None	 	  
 */	
#if defined (OS_NUCLEUS)
static void	RFResultCode_Task (UINT32 argc, void * argv)	{
#elif defined (OS_WINCE)
static DWORD RFResultCode_Task (void * pContext)	{
#endif
static void RFResultCode_Task(UINT32 param) {
	UINT8	AnyDeviceResultCodeTimeoutOccuredFlag;	
	UINT16	i;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst;	
		
	RFIntHostApp.pF_StartupWait(HPRF_STARTUP_RESULTCODE_READY);

	//	Wait for RF Stack to be initialized
	while (!RFReceive_IsRFStackInitialized())	{	RFIntHostApp.pF_HwDelayMsec(100);	}

#if defined (OS_WINCE)
	RFIntHostApp.pF_ErrorPrintf("\r\nRFResultCode_Task: Running!");
#endif
	while (1)	{
		
		pdst = g_pRFDeviceDescriptiveTable;
		AnyDeviceResultCodeTimeoutOccuredFlag = FALSE;
		
		for (i=0; i<g_MaxDescriptiveTableEntries; i++, pdst++)	{
			if ((pdst->linkstate != RF_DEVICE_LINK_STATE_EMPTY) || (i == RFDesc_GetASPDescriptiveTableIndex()) 
																|| (i == RFDesc_GetMLMEDescriptiveTableIndex()))	{
				if (pdst->ResultCodeTimer)	{
					if (--pdst->ResultCodeTimer == 0)	{
						pdst->ResultCodeTimedOut = TRUE;
						AnyDeviceResultCodeTimeoutOccuredFlag = TRUE;
						pdst->COMLog.txpacketresultcodetimeouts += 1;
					}	
				}
			}		
		}

		if (AnyDeviceResultCodeTimeoutOccuredFlag)	{
			RFReceive_SetResultCodeTimeoutEvent();	
		} 	
		
		RFIntHostApp.pF_HwDelayMsec(RF_RESULT_CODE_TASK_RATE);
	}
}

////////////////////////////////////////////////////////////////////////////////
//	GLOBAL FUNCTIONS 
///////////////////////////////////////////////////////////////////////////////

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function reloads the rf device result code timer counter	 	   
 *
 * \detail		
 *
 * \date        10/11/2006
 *
 * \param	    index 		
 *
 * \return		None	 				
 *
 * \retval		None	 	  
 */	
void	RFResultCode_ReTriggerTimer (UINT8 index)	{	
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);
		
	pdst->ResultCodeTimedOut = FALSE;
	pdst->ResultCodeTimer    = RFReceive_GetLocalDeviceResultCodeTimeout();
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function processes the rf device result code returned by the rf stack after a packet has been transmitted 	   
 *
 * \detail		The result code is anaylized, in case of ACK event flags are set to inform the rf transmit task to possibly submit the
 *				next packet, in case of NAK, the retry delay is started if programed and when the retry delay expires the rf transmit
 *				task is informed. 
 *	
 *
 * \date        10/11/2006
 *
 * \param	    None		
 *
 * \return		UINT16	resultcode	 				
 *
 * \retval		SUCCESS/FAILURE		 	  
 */	
UINT16	RFResultCode_MCPProcessing (UINT8 index, UINT8 resultcode)	{
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = (g_pRFDeviceDescriptiveTable+index);
	//TYPE_RF_ACTIVE_SCAN_REPLY_PACKET_DATA *pactivescan;
	UINT16 result = SUCCESS;
	BOOL	 isTraceEnabled;
	//UNSIGNED pipesize, avaliable, messages;
	BOOL	allpktsexpired;

	RFIntHostApp.pF_IsRFRxDebugTraceEnabled(&isTraceEnabled);

	//	Clear retry parameters
	pdst->RetryTimedOut = FALSE;
	pdst->RetryTimer    = 0;
				
	//	Clear result code parameters
	pdst->ResultCodeTimedOut = FALSE;
	pdst->ResultCodeTimer 	 = 0;

	//	Store result code
	pdst->resultcode		 = resultcode;

	switch (resultcode)	{
		case	gSuccess_c:
			//	make call back to the cmd initiator func
			RFTransmit_MCPPacketResultCodeCallBack (index);
			//	Result code is ACK				
			if (pdst->linkstate == RF_DEVICE_LINK_STATE_TEMP_CONN_ACCESS)	{
				if (SUCCESS != RFDesc_ReleaseDescriptiveEntryIfPipeIsEmpty(index))	{
					pdst->resultcodeavailable = TRUE;
					RFTransmit_TriggerRFTransmitTaskEvent(index);
				}
			}	else	{				
				pdst->resultcodeavailable = TRUE;
				RFTransmit_TriggerRFTransmitTaskEvent(index);
				RFHeartBeat_ReTriggerTimer (index);
			}
			//	turn on the received packet indicator
			RFIntHostApp.pF_RFReceiverIndicatorFlash();
			pdst->COMLog.rxpacketacks++;
			pdst->COMLog.txpacketturnaroundtime = OS_RetrieveClock() - pdst->COMLog.txpacketposttime;	
			pdst->COMLog.txpacketaccturnaroundtime += pdst->COMLog.txpacketturnaroundtime;	
		#ifdef	RF_DBG_LOG
			RFComLog_LogEvent (RF_DBG_MCP_RX_CONF_PACKET, index, gMcpsDataCnf_c, 0, resultcode, pdst->COMLog.rxpacketacks, OS_RetrieveClock());
		#endif	
			break;

		case	gNoAck_c:
		case	gTransactionOverflow_c:
		case	gTransactionExpired_c:
		case	gChannelAccessFailure_c:				
			//	increase packet retry count
//		    trace_printk(KERN_INFO "%s --- failure type = %i\n", __FUNCTION__, resultcode);
			pdst->COMLog.txpacketretrycount += 1;
			if (pdst->txpacketretrycntr >= pdst->maxtxpacketretries)	{	
				//	all retries failed or retries are disabled
				//	make call back to the cmd initiator func
				RFTransmit_MCPPacketResultCodeCallBack (index);
				//	set device link state to fail
				RFDesc_LinkStateTransition (index, RF_DEVICE_LINK_STATE_FAIL);		
		#ifndef	RF_NETWORK_MASTER_DEVICE			
				if (index == DEVICE_DESCRIPTIVE_TABLE_INDEX)	{
					//	On Slave devices infrom the transmit task and start the link request process		
					RFDesc_LinkStateTransition (index, RF_DEVICE_LINK_STATE_CONNECTING);				
					RFAPP_ConnectionFailedwithGateway();	
				}
		#endif		
			}	else if (RFReceive_GetLocalDeviceRetryDelayAtRetryCount((UINT8)pdst->txpacketretrycntr))	{
				//	program retry delay
				pdst->RetryTimer = RFReceive_GetLocalDeviceRetryDelayAtRetryCount((UINT8)pdst->txpacketretrycntr);
			}	else	{
				//	no retry delay programed, request retransmission of packet 

				//	check for expired packets, make call back only for expired packets 
				RFTransmit_MCPPacketTimeToLiveExpirationChk (index, &allpktsexpired);				
				
				//	set result code is avaliable indicators for the transmit task
				pdst->resultcodeavailable = TRUE;
				RFTransmit_TriggerRFTransmitTaskEvent(index);
			}
			pdst->COMLog.rxpacketnaks++;			
		#ifdef	RF_DBG_LOG
			RFComLog_LogEvent (RF_DBG_MCP_RX_CONF_PACKET, index, gMcpsDataCnf_c, 0, resultcode, pdst->COMLog.rxpacketnaks, OS_RetrieveClock());
		#endif	
			break;
		
		case	gInvalidGts_c:
		case	gUnavailableKey_c:
		case	gInvalidParameter_c:
		case	gFrameTooLong_c:
		case	gFailedSecurityCheck_c:
		default:		
			//	packet rejected by SMAC, packet is invalid 
			//	make call back to the cmd initiator func 
			RFTransmit_MCPPacketResultCodeCallBack (index);
			//	increase packet fail/error count
			pdst->COMLog.txpacketerrorcount += 1;
			pdst->resultcodeavailable = TRUE;
			RFTransmit_TriggerRFTransmitTaskEvent(index);
			
		#ifdef	RF_DBG_LOG
			RFComLog_LogEvent (RF_DBG_MCP_RX_CONF_PACKET, index, gMcpsDataCnf_c, 0, resultcode, pdst->COMLog.txpacketerrorcount, OS_RetrieveClock());
		#endif	
			break;
	}

	RFTransmit_RFStackBufferAvailableEvent();

	return (result);
}

 /**
 *
 * \author      Hazrat Shah	
 *
 * \brief		The active/passive/energy scan result code processing   
 *
 * \detail		
 *
 * \date        10/10/2006
 *
 * \param	    UINT8 index, nwkMessage_t* pMsg		
 *
 * \return		UINT16	 				
 *
 * \retval		SUCCESS/FAILURE	 	  
 */	
void HPRFGW_ProcessScanResult(UINT8 index, nwkMessage_t* pMsg)
{
	//UINT8 n, i;
	TYPE_RFAPP_DATA_BUF	   rfapp;
	TYPE_RF_ACTIVE_SCAN_REPLY_PACKET_DATA *pactivescan;	
	

	switch (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.req)	{	
		//	Process scan for other device on the network
		case  gScanModeActive_c:
			//	Active scan result: Forward packet to rf application task
			memset((UINT8*) &rfapp,	0, sizeof(rfapp.inthdr)+sizeof(rfapp.pkthdr));

			pactivescan = (TYPE_RF_ACTIVE_SCAN_REPLY_PACKET_DATA*) (&rfapp.data);
			rfapp.pkthdr.length  = sizeof(TYPE_INTERNAL_RF_PACKET_HEADER)-1 + sizeof(pactivescan->noofdescriptors) + sizeof(panDescriptor_t)*pMsg->msgData.scanCnf.resultListSize; 
			rfapp.pkthdr.type    = RF_PACKET_ACTIVE_SCAN_REPLY;
			pactivescan->noofdescriptors = pMsg->msgData.scanCnf.resultListSize;
			if (pMsg->msgData.scanCnf.resultListSize)	{
				memcpy((UINT8*)(&pactivescan->pandesc), (UINT8*) (pMsg->msgData.scanCnf.resList.pPanDescriptorList), sizeof(panDescriptor_t)*pMsg->msgData.scanCnf.resultListSize);
			}
			RFAPP_PostToPipe ((UINT8 *)(&rfapp), sizeof(rfapp.inthdr)+rfapp.pkthdr.length+1);		
			//	Make call back

			//	Release memory
			RFReceive_DisableTimerInterrupt ();
			MSG_Free(pMsg->msgData.scanCnf.resList.pPanDescriptorList);
			RFReceive_EnableTimerInterrupt ();
			break;
		
		case  gScanModePassive_c: 
		case  gScanModeOrphan_c: 
			break;

		case gScanModeED_c:
			//	Energy scan result: Forward packet to rf application task
			memset((UINT8*) &rfapp,	0, sizeof(rfapp.inthdr)+sizeof(rfapp.pkthdr));
			
			rfapp.pkthdr.length = sizeof(TYPE_INTERNAL_RF_PACKET_HEADER)-1 + sizeof(TYPE_RF_ENERGY_SCAN_REPLY_PACKET_DATA); 
			rfapp.pkthdr.type   = RF_PACKET_ENERGY_SCAN_REPLY;
			memcpy((UINT8*)(&rfapp.data), (UINT8*)(&pMsg->msgData.scanCnf.resList.pEnergyDetectList), sizeof(TYPE_RF_ENERGY_SCAN_REPLY_PACKET_DATA));			
			RFAPP_PostToPipe ((UINT8 *) &rfapp, sizeof(rfapp.inthdr)+rfapp.pkthdr.length+1);
			//	The list of detected energies must be freed. 
			RFReceive_DisableTimerInterrupt ();
			MSG_Free(pMsg->msgData.scanCnf.resList.pEnergyDetectList);
			RFReceive_EnableTimerInterrupt ();
			break;
	} 
}

 /**
 *
 * \author      Hazrat Shah	
 *
 * \brief		The MLME result code processing   
 *
 * \detail		
 *
 * \date        10/10/2006
 *
 * \param	    UINT8 index, nwkMessage_t* pMsg		
 *
 * \return		UINT16	 				
 *
 * \retval		SUCCESS/FAILURE	 	  
 */	
UINT16 RFResultCode_MLMEProcessing (UINT8 index, nwkMessage_t* pMsg)	{
	UINT16	result = SUCCESS;
	UINT8	resultcode = 0; 
	UINT8   resultcodeavailable = FALSE;
	BOOL	isTraceEnabled;

	RFIntHostApp.pF_IsRFRxDebugTraceEnabled(&isTraceEnabled);

	switch(pMsg->msgType)	{
		case	gNwkAssociateCnf_c:
			if (isTraceEnabled)	{
				RFIntHostApp.pF_ConsolePrintf ("Association confirmed: addr[%02x%02x] status[%02x]\r\n", pMsg->msgData.associateCnf.assocShortAddress[0], pMsg->msgData.associateCnf.assocShortAddress[1], pMsg->msgData.associateCnf.status);
			}
			//if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gMlmeAssociateReq_c)		{	
				resultcode          = pMsg->msgData.associateCnf.status;
				resultcodeavailable = TRUE;		
			//}
			break;
		case	gNwkDisassociateCnf_c:  		
			if (isTraceEnabled)	{
				RFIntHostApp.pF_ConsolePrintf ("DisAssociation confirmed: status[%02x]r\n", pMsg->msgData.disassociateCnf.status);
			}
			//if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gMlmeDisassociateReq_c)		{	
				resultcode          = pMsg->msgData.disassociateCnf.status;
				resultcodeavailable = TRUE;		
			//}
			break;
		case	gNwkScanCnf_c:
			//if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gMlmeScanReq_c)				{	
				resultcode          = pMsg->msgData.scanCnf.status;
				resultcodeavailable = TRUE;		
				HPRFGW_ProcessScanResult(index, pMsg);	
			//}
			break;		
		
		case	gNoBeacon_c:
			if (isTraceEnabled)	{
				RFIntHostApp.pF_ConsolePrintf ("NoBeacon MLME confirmation message type[%02x] index[%02x]\r\n", pMsg->msgType, index);
			}
			resultcode = gSuccess_c;
			resultcodeavailable = TRUE;
			break;
		
		case	gNwkGtsCnf_c:				
		case	gNwkGetCnf_c:				
		case	gNwkResetCnf_c:
		case	gNwkRxEnableCnf_c:
		case	gNwkSetCnf_c:
		case	gNwkStartCnf_c:
		case	gNwkPollCnf_c:
		case	gNwkErrorCnf_c:
		case	gNwkMaintenanceScanCnf_c:       
			if (isTraceEnabled)	{
				RFIntHostApp.pF_ConsolePrintf ("Unexpected MLME confirmation message type %02x, index %02x\r\n", pMsg->msgType, index);
			}			
			//	Log Error
			RFIntHostApp.pF_ErrorPrintf ("Unexpected MLME confirmation message type %02x, index %02x\r\n", pMsg->msgType, index);
			//	Log error at given descriptive entry
			resultcode = gSuccess_c;
			resultcodeavailable = TRUE;
			break;
		default:
			if (isTraceEnabled)	{
				RFIntHostApp.pF_ConsolePrintf ("Unrecognized MLME confirmation message type %02x, index %02x\r\n", pMsg->msgType, index);
			}
			//	Log Error
			RFIntHostApp.pF_ErrorPrintf ("Unrecognized MLME confirmation message type %02x, index %02x\r\n", pMsg->msgType, index);
			//	Log error at given descriptive entry
			result = FAILURE;		
			resultcode = gSuccess_c;
			resultcodeavailable = TRUE;
			break;
	}

	if (resultcodeavailable) { 
		if (resultcode != gSuccess_c) {
			if (isTraceEnabled)	{
				RFIntHostApp.pF_ConsolePrintf ("MLME failed result code[%02x]\r\n", resultcode);
			}
		}	
		//	Clear retry parameters
		g_pRFDeviceDescriptiveTable[index].RetryTimedOut = FALSE;
		g_pRFDeviceDescriptiveTable[index].RetryTimer    = 0;				
		//	Clear result code parameters
		g_pRFDeviceDescriptiveTable[index].ResultCodeTimedOut  = FALSE;
		g_pRFDeviceDescriptiveTable[index].ResultCodeTimer 	   = 0;		
		//	Set result code regs
		g_pRFDeviceDescriptiveTable[index].resultcode          = resultcode;	
		//	Inform MLME requester with result code 
		//	RFTransmit_MLMEPacketResultCodeCallBack (index);		
		g_pRFDeviceDescriptiveTable[index].resultcodeavailable = resultcodeavailable;	
		RFTransmit_TriggerRFTransmitTaskEvent(index);
	}	else	{
		//	mlme retry - not supported
		
	}

	return (result);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		The ASP result code processing   
 *
 * \detail		
 *
 * \date        10/10/2006
 *
 * \param	    UINT8 index, UINT8 MsgType		
 *
 * \return		UINT16	 				
 *
 * \retval		SUCCESS/FAILURE	 	  
 */	
UINT16 RFResultCode_ASPProcessing (UINT8 index, UINT8 MsgType)	{	
	UINT16 result  = SUCCESS;
	BOOL   confirm = FALSE;
	
	//	Clear retry parameters
	g_pRFDeviceDescriptiveTable[index].RetryTimedOut = FALSE;
	g_pRFDeviceDescriptiveTable[index].RetryTimer    = 0;				
	//	Clear result code parameters
	g_pRFDeviceDescriptiveTable[index].ResultCodeTimedOut = FALSE;
	g_pRFDeviceDescriptiveTable[index].ResultCodeTimer 	 = 0;

	switch(MsgType)	{
		case  gAspAppGetTimeCfm_c: 
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspGetTimeReq_c)			{	confirm = TRUE;		}
			break;
		case  gAspAppGetInactiveTimeCfm_c:
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspGetInactiveTimeReq_c)	{	confirm = TRUE;		}
			break;
		case  gAspAppDozeCfm_c:
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspDozeReq_c)				{	confirm = TRUE;		}
			break;
		case  gAspAppAutoDozeCfm_c:	
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspAutoDozeReq_c)			{	confirm = TRUE;		}
			break;
		case  gAspAppHibernateCfm_c:	
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspHibernateReq_c)			{	confirm = TRUE;		}
			break;
		case  gAspAppWakeCfm_c:
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspWakeReq_c)				{	confirm = TRUE;		}
			break;
		case  gAspAppEventCfm_c:          
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspEventReq_c)				{	confirm = TRUE;		}
			break;
		case  gAspAppTrimCfm_c:            
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspTrimReq_c)				{	confirm = TRUE;		}
			break;
		case  gAspAppDdrCfm_c:             
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspDdrReq_c)				{	confirm = TRUE;		}
			break;
		case  gAspAppPortCfm_c:            
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspPortReq_c)				{	confirm = TRUE;		}
			break;
		case  gAspAppClkoCfm_c:            
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspClkoReq_c)				{	confirm = TRUE;		}
			break;
		case  gAspAppTempCfm_c:            
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspTempReq_c)				{	confirm = TRUE;		}
			break;
		case  gAspAppNvRamCfm_c:           
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspNvRamReq_c)				{	confirm = TRUE;		}
			break;
		case  gAspAppBatteryCfm_c:         
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspBatteryReq_c)			{	confirm = TRUE;		}
			break;
		case  gAspAppSetNotifyCfm_c:       
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspSetNotifyReq_c)			{	confirm = TRUE;		}
			break;
		case  gAspAppSetMinDozeTimeCfm_c:  
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspSetMinDozeTimeReq_c)		{	confirm = TRUE;		}
			break;
		case  gAspAppTelecTestCfm_c:       
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspTelecTestReq_c)			{	confirm = TRUE;		}
			break;
		case  gAspAppTelecSetChannelCfm_c: 
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspTelecSetChannelReq_c)	{	confirm = TRUE;		}
			break;
		case  gAspAppSetPowerLevelCfm_c:   
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspSetPowerLevelReq_c)		{	confirm = TRUE;		}
			break;
		case  gAspAppAcomaCfm_c:           
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspAcomaReq_c)				{	confirm = TRUE;		}
			break;
		case  gAspAppGetMacStateCfm_c:     
			if (g_pRFDeviceDescriptiveTable[index].TxBuf.Transmit.inthdr.subtype == gAppAspGetMacStateReq_c)		{	confirm = TRUE;		}
			break;
		case  gAspErrorCfm_c:       
			//	Error occured
			break;
		default:
			result = FAILURE;
			break;
	}
	
	if (g_pRFDeviceDescriptiveTable[index].waitingforACK)	{
		if (confirm)	{
			g_pRFDeviceDescriptiveTable[index].resultcode			= gSuccess_c;
			//	Make ASP call-back
			g_pRFDeviceDescriptiveTable[index].resultcodeavailable	= TRUE;			
			RFTransmit_TriggerRFTransmitTaskEvent(index);
		}	else	{
			//	Unexpected reply
			g_pRFDeviceDescriptiveTable[index].resultcode			= gNoAck_c;
			//	HS_T Log incorrect response
			
			//	HS_T retry may be optional based on stack behavior and failure rate (this is a serious internal stack failure)
			if (g_pRFDeviceDescriptiveTable[index].txpacketretrycntr > g_pRFDeviceDescriptiveTable[index].maxtxpacketretries)	{	
				//	Make ASP call-back

				//	Packet retried maximum times, propor response was not received
				//	HS_T Log error ASP packet was not confirmed (expected confirmation not same what was received)
				g_pRFDeviceDescriptiveTable[index].resultcodeavailable = TRUE;
				RFTransmit_TriggerRFTransmitTaskEvent(index);
		#ifdef	ASP_MSG_RETRY_DELAY_ENABLE
			}	else if (RFReceive_GetTxPacketRetryDelay(g_pRFDeviceDescriptiveTable[index].txpacketretrycntr))	{
				//	Wait and then try to resend the packet 
				g_pRFDeviceDescriptiveTable[index].RetryTimer   	   = RFReceive_GetTxPacketRetryDelay(g_pRFDeviceDescriptiveTable[index].txpacketretry);
		#endif
			}	else	{
				//	Make ASP call-back 
				
				//	Retry is disabled, inform the transmit task  
				g_pRFDeviceDescriptiveTable[index].resultcodeavailable = TRUE;
				RFTransmit_TriggerRFTransmitTaskEvent(index);
			}
		}		
	}

	return (result);
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		This function processes the RF device result code timeouts 	   
 *
 * \detail		All RF devices are checked for timeouts and if any device's result code timedout and error message is logged and
 *				the system is reset
 *	
 *
 * \date        10/11/2006
 *
 * \param	    None		
 *
 * \return		None	 				
 *
 * \retval		None	 	  
 */	
void	RFResultCode_TimeoutProcessing (void)	{	
	UINT16	i;
	TYPE_RF_DESCRIPTIVE_TABLE *pdst = g_pRFDeviceDescriptiveTable;
	INT8	buf[200];
	UINT32	currNumResets, maxwait;

	for (i=0; i<g_MaxDescriptiveTableEntries; i++, pdst++)	{
		if (pdst->ResultCodeTimedOut)	{
			if			(i == RFDesc_GetMLMEDescriptiveTableIndex())	{	
				//	log error
				sprintf (buf, "Error: Result Code timeout occured on MLME packet type=%02x, packet subtype=%02x ", pdst->TxBuf.Transmit.inthdr.type, pdst->TxBuf.Transmit.inthdr.subtype);

#ifdef	RF_DBG_LOG
				RFComLog_LogEvent (RF_DBG_MLME_RESULT_TO_PACKET, i, g_pRFDeviceDescriptiveTable[i].TxBuf.Transmit.inthdr.type, g_pRFDeviceDescriptiveTable[i].TxBuf.Transmit.inthdr.subtype, 0, 0, OS_RetrieveClock());
#endif
			
			}	else if	(i == RFDesc_GetASPDescriptiveTableIndex())	{	
				//	log error
				sprintf (buf, "Error: Result Code timeout occured on ASP packet type=%02x, packet subtype=%02x ",  pdst->TxBuf.Transmit.inthdr.type, pdst->TxBuf.Transmit.inthdr.subtype);
#ifdef	RF_DBG_LOG
				RFComLog_LogEvent (RF_DBG_ASP_RESULT_TO_PACKET, i, g_pRFDeviceDescriptiveTable[i].TxBuf.Transmit.inthdr.type, g_pRFDeviceDescriptiveTable[i].TxBuf.Transmit.inthdr.subtype, 0, 0, OS_RetrieveClock());
#endif		
			}	else	{
				//	log error
				sprintf (buf, "Error: Result Code timeout occured on device %02x: %s", pdst->IDInfo.rfid, (INT8 *) &pdst->IDInfo.idstring);
#ifdef	RF_DBG_LOG
				RFComLog_LogEvent (RF_DBG_MCP_RESULT_TO_PACKET, i, 0, 0, g_pRFDeviceDescriptiveTable[i].TxBuf.Transmit.inthdr.type, 0, OS_RetrieveClock());
#endif	
			}		
			//	9/20/07 HS - removed to avoid filling up the error log, use the result code counter to 
			//				 track this error condition
			//RFIntHostApp.pF_ErrorPrintf(buf);			

#ifdef	RF_DBG_LOG
	//RFDebugLogShow (400);
#endif

#ifdef	RF_RESET_PIPE_ON_RESULT_CODE_TIMEOUT_ENABLED 
			// MNT - 7/5/2007 - Grab the reset count and wait till reset count
			// increments before setting the transmit event
			currNumResets = Rfdiags_getLowLevelResets();
				
			RFTransmit_BuildResetMACMessageAndPostToPipe(FALSE);
			
			//	HS - 11/06/07 - wait a maximum of 1-second for SMAC to finish the reset cycle
			//					the SMAC increments the Rfdiags_LowLevelResets cntr when the reset
			//					is done.
#if 0
			maxwait = TICKS_PER_SECOND; 
			while ((Rfdiags_LowLevelResets == currNumResets) && maxwait--) {
				OS_Sleep(1);
			} 
#endif
			//TODO: FIXME:
			maxwait = 1000 / 10; // The number of 10ms intervals in a second
			while( (Rfdiags_getLowLevelResets() == currNumResets) && maxwait--) {
				HwDelayMsec(10);
			}

			pdst->resultcode = gTransactionExpired_c;
			pdst->resultcodeavailable = TRUE;	

			RFTransmit_TriggerRFTransmitTaskEvent((UINT8)i);													
#else		
			RFIntHostApp.pF_HwDelayMsec(2000);			
			//	reset system
			ErrorSoftwareReset();
#endif
		}
	}
}

/**
 *
 * \author      Hazrat Shah	
 *
 * \brief		The Result Code init function  
 *
 * \detail		This function creates the rf result code timer 
 *
 * \date        10/10/2006
 *
 * \param	    None		
 *
 * \return		None	 				
 *
 * \retval		None	 	  
 */	
void	RFResultCode_Init (void)
{
#if defined (OS_NUCLEUS)
	//	Create rf result code task	 
    if(NU_Create_Task(&RFResultCodeTask.task, "TRFRESUL", RFResultCode_Task, 0, NU_NULL,
        RFResultCodeTask.stack, sizeof(RFResultCodeTask.stack), RF_RESULT_CODE_TASK_PRIORITY, 0, NU_PREEMPT, NU_START ) != NU_SUCCESS)	{
        RFIntHostApp.pF_ErrorHandler("TRFRESUL: Could not create task!");
    }
#elif defined (OS_WINCE)
	RFResultCodeTask.task = CreateThread(NULL, 0, RFResultCode_Task, pHPRDevice, 0, NULL);
	if (NULL == RFResultCodeTask.task)
	{
		RFIntHostApp.pF_ErrorPrintf("TRFRESUL: Could not create task!");
	}
	// Set thread priority
	CeSetThreadPriority(RFResultCodeTask.task, RF_RESULT_CODE_TASK_PRIORITY);
#endif
	RFResultCodeTask.task = OsCreateNamedAdvTaskPri(RFResultCode_Task, 0, 0, RF_RESULT_CODE_TASK_PRIORITY, "TRFRESUL");
	if (!RFResultCodeTask.task)
	{
		RFIntHostApp.pF_ErrorPrintf("TRFRESUL: Fail create task!");
	}
}
